<?php
/**
 * ViciDial Integration Example with Speaker Options
 * 
 * This example shows how to integrate the Vici WebRTC phone with speaker options
 * into a ViciDial environment.
 */

// ViciDial database connection (adjust these values for your environment)
$db_host = 'localhost';
$db_user = 'cron';
$db_pass = '1234';
$db_name = 'asterisk';

// Function to get phone settings from ViciDial
function getPhoneSettings($phone_login) {
    global $db_host, $db_user, $db_pass, $db_name;
    
    try {
        $pdo = new PDO("mysql:host=$db_host;dbname=$db_name", $db_user, $db_pass);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Get phone settings from vicidial_phones table
        $stmt = $pdo->prepare("SELECT phone_login, phone_pass, server_ip, webphone_options FROM vicidial_phones WHERE phone_login = ?");
        $stmt->execute([$phone_login]);
        $phone = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $phone;
    } catch (PDOException $e) {
        error_log("Database error: " . $e->getMessage());
        return false;
    }
}

// Function to build options string with speaker configuration
function buildOptionsString($base_options, $speaker_config = 'BOTH') {
    $options = [];
    
    // Add existing options
    if (!empty($base_options)) {
        $options[] = $base_options;
    }
    
    // Add speaker configuration
    $options[] = "SPEAKER" . $speaker_config;
    
    return implode('--', $options);
}

// Get phone login from request
$phone_login = $_GET['phone_login'] ?? $_POST['phone_login'] ?? '';

if (empty($phone_login)) {
    die("Error: Phone login required");
}

// Get phone settings from database
$phone_settings = getPhoneSettings($phone_login);

if (!$phone_settings) {
    die("Error: Phone not found or database connection failed");
}

// Get speaker configuration from request (default to BOTH)
$speaker_config = $_GET['speaker'] ?? $_POST['speaker'] ?? 'BOTH';

// Validate speaker configuration
if (!in_array($speaker_config, ['LEFT', 'RIGHT', 'BOTH'])) {
    $speaker_config = 'BOTH';
}

// Build options string
$options = buildOptionsString($phone_settings['webphone_options'], $speaker_config);

// Encode parameters for viciphone.php
$encoded_login = base64_encode($phone_settings['phone_login']);
$encoded_pass = base64_encode($phone_settings['phone_pass']);
$encoded_server = base64_encode($phone_settings['server_ip']);
$encoded_options = base64_encode($options);

// Build viciphone URL
$viciphone_url = "viciphone.php?phone_login=$encoded_login&phone_pass=$encoded_pass&server_ip=$encoded_server&options=$encoded_options";
?>

<!DOCTYPE html>
<html>
<head>
    <title>ViciDial WebRTC Phone - Speaker Options</title>
    <style>
        body { 
            font-family: Arial, sans-serif; 
            margin: 20px; 
            background-color: #f5f5f5;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .phone-info {
            background: #e8f4fd;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .speaker-controls {
            background: #f9f9f9;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .phone-frame {
            border: 2px solid #ddd;
            border-radius: 8px;
            overflow: hidden;
            margin: 20px 0;
        }
        .speaker-btn {
            background: #007bff;
            color: white;
            border: none;
            padding: 8px 16px;
            margin: 5px;
            border-radius: 4px;
            cursor: pointer;
        }
        .speaker-btn:hover {
            background: #0056b3;
        }
        .speaker-btn.active {
            background: #28a745;
        }
        .info-table {
            width: 100%;
            border-collapse: collapse;
            margin: 10px 0;
        }
        .info-table td {
            padding: 8px;
            border-bottom: 1px solid #ddd;
        }
        .info-table td:first-child {
            font-weight: bold;
            width: 150px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>ViciDial WebRTC Phone - Speaker Options</h1>
        
        <div class="phone-info">
            <h3>Phone Information</h3>
            <table class="info-table">
                <tr>
                    <td>Phone Login:</td>
                    <td><?php echo htmlspecialchars($phone_settings['phone_login']); ?></td>
                </tr>
                <tr>
                    <td>Server IP:</td>
                    <td><?php echo htmlspecialchars($phone_settings['server_ip']); ?></td>
                </tr>
                <tr>
                    <td>Base Options:</td>
                    <td><?php echo htmlspecialchars($phone_settings['webphone_options'] ?? 'None'); ?></td>
                </tr>
                <tr>
                    <td>Current Speaker:</td>
                    <td><?php echo htmlspecialchars($speaker_config); ?></td>
                </tr>
            </table>
        </div>
        
        <div class="speaker-controls">
            <h3>Speaker Configuration</h3>
            <p>Select which speaker(s) to use for audio output:</p>
            
            <button class="speaker-btn <?php echo $speaker_config === 'BOTH' ? 'active' : ''; ?>" 
                    onclick="changeSpeaker('BOTH')">Both Speakers</button>
            <button class="speaker-btn <?php echo $speaker_config === 'LEFT' ? 'active' : ''; ?>" 
                    onclick="changeSpeaker('LEFT')">Left Speaker Only</button>
            <button class="speaker-btn <?php echo $speaker_config === 'RIGHT' ? 'active' : ''; ?>" 
                    onclick="changeSpeaker('RIGHT')">Right Speaker Only</button>
            
            <p><small>You can also change the speaker setting using the dropdown in the phone interface.</small></p>
        </div>
        
        <div class="phone-frame">
            <iframe id="viciphone-frame" 
                    src="<?php echo htmlspecialchars($viciphone_url); ?>" 
                    width="424" height="260" 
                    frameborder="0"
                    allow="microphone; camera">
            </iframe>
        </div>
        
        <div style="margin-top: 20px;">
            <h3>Integration Notes</h3>
            <ul>
                <li>This example shows how to integrate speaker options with ViciDial database</li>
                <li>Speaker configuration can be set via URL parameter or dynamically changed</li>
                <li>The phone interface includes a dropdown for real-time speaker selection</li>
                <li>All existing ViciDial phone options are preserved and can be combined</li>
            </ul>
        </div>
    </div>

    <script>
        function changeSpeaker(speaker) {
            // Update URL with new speaker configuration
            const url = new URL(window.location);
            url.searchParams.set('speaker', speaker);
            window.location.href = url.toString();
        }
        
        // Add some interactivity
        document.addEventListener('DOMContentLoaded', function() {
            console.log('ViciDial WebRTC Phone with Speaker Options loaded');
            console.log('Current speaker configuration:', '<?php echo $speaker_config; ?>');
        });
    </script>
</body>
</html> 